# Element Pack Custom Fonts

A self-hosted custom fonts manager for Element Pack that allows users to upload and use their own font files in Elementor.

## Features

- ✅ Upload custom font files (WOFF2, WOFF, TTF, EOT, SVG)
- ✅ Multiple font variations (weights & styles)
- ✅ Auto-generate @font-face CSS
- ✅ Seamless Elementor integration
- ✅ Real-time editor preview
- ✅ RTL support
- ✅ Security validation

## File Structure

```
custom-fonts/
├── init.php                              # Feature initializer
├── class-ep-custom-fonts-manager.php     # Main manager (CPT, Elementor integration)
├── class-ep-custom-fonts-admin.php       # Admin UI & meta boxes
├── class-ep-font-face-generator.php      # CSS generator
├── class-ep-font-uploader.php            # Upload handler & validation
├── assets/
│   ├── css/
│   │   ├── admin.css                     # Admin styles
│   │   └── admin.rtl.css                 # RTL styles (auto-generated)
│   └── js/
│       ├── admin.js                      # Admin interface
│       └── editor.js                     # Elementor editor integration
└── README.md                             # This file
```

## Usage

### Adding a Custom Font

1. Go to **Element Pack > Custom Fonts**
2. Click **Add New**
3. Enter the font name (e.g., "My Custom Font")
4. Upload font files for each variation:
   - **WOFF2**: Best compression (recommended)
   - **WOFF**: Wide browser support
   - **TTF**: Safari, iOS, Android
   - **EOT**: Internet Explorer
   - **SVG**: Legacy iOS
5. Select font weight (100-900) and style (normal/italic/oblique)
6. Click **Add Font Variation** for additional weights/styles
7. Click **Publish**

### Using Custom Fonts in Elementor

1. Edit any page with Elementor
2. Select a widget with typography controls
3. In the typography section, click the **Font Family** dropdown
4. Your custom font will appear under **"Element Pack Fonts"**
5. Select it and see it applied in real-time

## Technical Details

### Custom Post Type

- **Slug**: `ep_custom_font`
- **Capability**: `manage_options`
- **Public**: `false` (admin only)

### Meta Keys

- `ep_font_files`: Array of font variations with file data
- `ep_font_face_css`: Cached @font-face CSS

### Supported Font Formats

| Format | MIME Types | Browser Support |
|--------|-----------|-----------------|
| WOFF2 | font/woff2 | Modern browsers |
| WOFF | font/woff, application/font-woff | All modern browsers |
| TTF | font/ttf, application/x-font-ttf | Safari, iOS, Android |
| EOT | application/vnd.ms-fontobject | IE 6-11 |
| SVG | image/svg+xml | Legacy iOS |

### Font Face Generation

Generated CSS example:
```css
@font-face {
    font-family: "My Custom Font";
    font-style: normal;
    font-weight: 400;
    font-display: swap;
    src: url('path/to/font.woff2') format('woff2'),
         url('path/to/font.woff') format('woff');
}
```

### Security

- File type validation (extension + MIME)
- SVG sanitization (script tag detection)
- Capability checks (`manage_options`)
- Nonce verification
- SQL injection prevention

### Performance

- CSS caching (regenerated only on save)
- Font caching (loaded once per editor session)
- Conditional loading (only used fonts)
- `font-display: swap` for better performance

## Hooks & Filters

### Filters

```php
// Modify generated @font-face CSS
add_filter('ep_custom_fonts_font_face_css', function($css, $font_family, $font_data) {
    // Your modifications
    return $css;
}, 10, 3);
```

### Actions

```php
// After font is saved
add_action('save_post_ep_custom_font', function($post_id) {
    // Your code
});

// Before font is deleted
add_action('before_delete_post', function($post_id) {
    if (get_post_type($post_id) === 'ep_custom_font') {
        // Your code
    }
});
```

## Browser Compatibility

- ✅ Chrome/Edge (latest)
- ✅ Firefox (latest)
- ✅ Safari (latest)
- ✅ iOS Safari
- ✅ Android Browser
- ✅ Internet Explorer 9+ (with EOT)

## Best Practices

1. **Always upload WOFF2** for best compression (60-80% smaller than TTF)
2. **Include WOFF** as fallback for older browsers
3. **Use semantic names** (e.g., "Brand Regular", not "font1")
4. **Limit variations** to what you actually need
5. **Test on multiple devices** before deploying

## Troubleshooting

### Font not appearing in Elementor

1. Clear Elementor cache (**Elementor > Tools > Regenerate CSS**)
2. Clear browser cache
3. Check if font post is published
4. Verify font files are uploaded correctly

### Font not loading on frontend

1. Check browser console for 404 errors
2. Verify file URLs are accessible
3. Regenerate font CSS (re-save the font)
4. Check MIME types are allowed on server

### Font looks different than expected

1. Verify correct font weight/style selected
2. Check if multiple @font-face rules conflict
3. Use browser DevTools to inspect applied font
4. Ensure font files are not corrupted

## Changelog

### Version 1.0.0
- Initial release
- Support for WOFF2, WOFF, TTF, EOT, SVG
- Multiple font variations
- Elementor integration
- Admin UI with file uploader
- RTL support
- Security validation

## Credits

Developed for Element Pack Pro by BdThemes
Inspired by Elementor Pro's custom fonts feature
